package gov.va.med.mhv.usermgmt.test.builder;

import java.sql.Timestamp;
import java.util.Date;

import junit.framework.Assert;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.ServiceResponse;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.delegate.InPersonAuthenticationServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.PatientServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.service.delegate.UserProfileServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.UserRegistrationServiceDelegate;
import gov.va.med.mhv.usermgmt.transfer.Facility;
import gov.va.med.mhv.usermgmt.transfer.FacilityInfo;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthentication;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;

/**
 * Utility class to support user testing.
 * Must call setUp() to register the user. A user with a random user account, 
 * if no user name is specified in the given UserProfile.
 * Must call tearDown() to remove the user.
 * @author Rob Proper
 *
 */
public final class UserBuilder {
	
	private static final Log LOG = LogFactory.getLog(UserBuilder.class);
	
    private final User user;

	public UserBuilder(User user) {
        Precondition.assertNotNull("user", user);
		this.user = user;
	}

	public UserProfile getUserProfile() { 
		return user.getUserProfile(); 
	}
	public void setUserProfile(UserProfile  userProfile) { 
		user.setUserProfile(userProfile); 
	}
	public String getPassword() { 
		return user.getPassword(); 
	}
	public String getUserName() { 
		return (getUserProfile() != null) ? 
            getUserProfile().getUserName() : null; 
	}
	public boolean isToBeIPAed() { 
		return user.isToBeIPAed(); 
	}

	public void setUp() throws Exception {
		Assert.assertNotNull("No user profile", getUserProfile());
		boolean isPatient = BooleanUtils.isTrue(getUserProfile().
			getIsPatient()); 
		LOG.info("Setup user '" + getUserProfile().getUserName() + "' as " 
			+ (isPatient ? (isToBeIPAed() ? "" : "non-") + "IPAed " : "")
			+ (isPatient ? "patient" : "non-patient"));
		if (StringUtils.isBlank(getUserName())) {
			getUserProfile().setUserName(UserProfileUtil.generateUserName());
		} else {
		    tearDown();
		}
		registerUser();
		updateFacilitiesIfNeeded();
		ensureUserIPAedIfNeeded();
		LOG.info("User has been successfully setup\n\n");
	}

	public void tearDown() throws Exception {
		LOG.debug("Tearing down existing user");
		removeUser();
    	UserProfileUtil.delete(find());
	}
	
	private void registerUser() {
		Assert.assertNotNull("No user profile", getUserProfile());
		LOG.debug("Registering user"); 
		UserRegistrationServiceDelegate delegate = 
			ServiceDelegateFactory.createUserRegistrationServiceDelegate();
		UserProfileServiceResponse response = delegate.registerUser(
			getUserProfile(), getUserProfile().getSsn(), getUserProfile().
			getUserName(), getPassword(), getPassword(), 
			new Timestamp(System.currentTimeMillis()));
		MessageHandler.handleMessages(response, response.getUserProfile(),
			"Unable to register user '" + getUserName() + "'"); 
		setUserProfile(response.getUserProfile());
		LOG.debug("User has been registered");
	}
	
	private void updateFacilitiesIfNeeded() {
		if (BooleanUtils.isTrue(getUserProfile().getIsPatient())) {
			ServiceResponse r = ServiceDelegateFactory.
				createPatientServiceDelegate().updatePatientFacilities(
					getUserProfile().getUserName());
			MessageHandler.handleMessages(r, 
				"Unable to update patient facilities");
			LOG.debug("Updated patient facilities");
		}
	}

	private void ensureUserIPAedIfNeeded() {	
		if ((getUserProfile().getIsPatient() == null)
			|| !getUserProfile().getIsPatient().booleanValue())
		{
			LOG.debug("User is not a patient");
			return;
		}
		if (!isToBeIPAed()) {
			LOG.debug("User not to be IPA-ed");
			return;
		}
		LOG.debug("IPA patient");
    	PatientServiceDelegate patientDelegate = ServiceDelegateFactory.
    		createPatientServiceDelegate();
    	PatientServiceResponse patientResponse = patientDelegate.
		getPatientForUser(getUserProfile());
		Patient patient = patientResponse.getPatient();
		Assert.assertNotNull("Patient record for user '" + getUserName() 
			+ "' not found", patient);
		
		InPersonAuthenticationServiceDelegate ipaDelegate = 
			ServiceDelegateFactory.createInPersonAuthenticationServiceDelegate();
		InPersonAuthenticationServiceResponse response = ipaDelegate.
			getAuthenticationForPatient(patient.getId());
		InPersonAuthentication ipa = response.getInPersonAuthentication();
		if ((ipa != null) && AuthenticationStatus.getEnum(
				AuthenticationStatus.AUTHENTICATED).equals(ipa.getStatus())) 
		{
			LOG.warn("Patient has already been IPA-ed");
			return;
		}
		java.util.Set facilities = patient.getFacilitys();
		if ((facilities == null) || (facilities.size() == 0)) {
			LOG.warn("Patient has no facilities");
			return;
		}
		Facility facility = (Facility) facilities.iterator().next();
		if (facility == null) {
			LOG.error("Failed to get facility");
			return;
		}
		FacilityInfo info = facility.getFacilityInfo();
		if (info == null) {
			LOG.error("Failed to get facility info for facility "
				+ facility.getName());
			return;
		}
		if (ipa == null) {
			ipa = TransferObjectFactory.createInPersonAuthentication();
		}
		ipa.setAuthenticationDate(new Timestamp(System.currentTimeMillis()));
		ipa.setApprovedForRecordsAccess(true);
		ipa.setIdentificationPresented(true);
		ipa.setVideoViewed(true);
		ipa.setPatient(patient);
		ipa.setAuthenticatingFacility(info);
		ipa.setParticipationFormSigned(java.lang.Boolean.TRUE);
		ServiceResponse r = ipaDelegate.authenticate(ipa);
		MessageHandler.handleMessages(r, "Unable to authenticate");
        // Do not wait for response
		r = ipaDelegate.finalizeAuthentication(ipa, null);
		MessageHandler.handleMessages(r, "Unable to finalize authentication");
		LOG.debug("Patient has been IPAed");
		return;
	}

	private void removeUser() {
		if (!StringUtils.isBlank(getUserName())) {
			ServiceResponse response = ServiceDelegateFactory.
				createUserServiceDelegate().removeUser(getUserName());
			MessageHandler.handleMessages(response, "Removing existing user '" 
				+ getUserName() + "'", LogLevel.Warn);
		}
	}

	public UserProfile find() {
		if (StringUtils.isBlank(getUserName())) {
			return null;
		}
		UserProfileServiceDelegate delegate = ServiceDelegateFactory.
			createUserProfileServiceDelegate();
		UserProfileServiceResponse response = delegate.getProfileForUser(
			getUserName());
		if (response.getMessages().hasErrorMessages()) {
			LOG.error("Finding user profile '" + getUserName() + "': "
				+ response.getMessages().getErrorMessages());
		}
		return response.getUserProfile();
	}


}